# jdbc-template
### 吴远韬 PB20061233
-----


## 表的实体用户类
```java
@Data
@Table(name = "orm_user")
public class User implements Serializable {
    /**
     * 主键
     */
    @Pk
    private Long id;

    /**
     * 用户名
     */
    private String name;
    
    
    // 省略....
}
```
## 用户数据访问接口类UserDao
```java
@Repository
public class UserDao extends BaseDao<User, Long> {

    @Autowired
    public UserDao(JdbcTemplate jdbcTemplate) {
        super(jdbcTemplate);
    }

    /**
     * 保存用户
     *
     * @param user 用户对象
     * @return 操作影响行数
     */
    public Integer insert(User user) {
        return super.insert(user, true);
    }

    /**
     * 根据主键删除用户
     *
     * @param id 主键id
     * @return 操作影响行数
     */
    public Integer delete(Long id) {
        return super.deleteById(id);
    }

    /**
     * 更新用户
     *
     * @param user 用户对象
     * @param id   主键id
     * @return 操作影响行数
     */
    public Integer update(User user, Long id) {
        return super.updateById(user, id, true);
    }

    /**
     * 根据主键获取用户
     *
     * @param id 主键id
     * @return id对应的用户
     */
    public User selectById(Long id) {
        return super.findOneById(id);
    }

    /**
     * 根据查询条件获取用户列表
     *
     * @param user 用户查询条件
     * @return 用户列表
     */
    public List<User> selectUserList(User user) {
        return super.findByExample(user);
    }
}
```

## jdbc访问接口对象基类 BaseDao
```java
@Slf4j
public class BaseDao<T, P> {
    private JdbcTemplate jdbcTemplate;
    private Class<T> clazz;
}
```
基类中包含方法的实现

以下分析insert的实现
```java
protected Integer insert(T t, Boolean ignoreNull) {
    String table = getTableName(t);

    List<Field> filterField = getField(t, ignoreNull);

    List<String> columnList = getColumns(filterField);

    String columns = StrUtil.join(Const.SEPARATOR_COMMA, columnList);

    // 构造占位符
    String params = StrUtil.repeatAndJoin("?", columnList.size(), Const.SEPARATOR_COMMA);

    // 构造值
    Object[] values = filterField.stream().map(field -> ReflectUtil.getFieldValue(t, field)).toArray();

    String sql = StrUtil.format("INSERT INTO {table} ({columns}) VALUES ({params})", Dict.create().set("table", table).set("columns", columns).set("params", params));
    log.debug("【执行SQL】SQL：{}", sql);
    log.debug("【执行SQL】参数：{}", JSONUtil.toJsonStr(values));
    return jdbcTemplate.update(sql, values);
}
```
首先调用`getTableName`获取User表的表名,然后调用`getField`和`getColumns`获取User表中的所有字段名。然后构造SQL语句，最后调用`jdbcTemplage.update()`更新数据库

## Service
User业务接口
```java
public interface IUserService {
    /**
     * 保存用户
     */
    Boolean save(User user);

    /**
     * 删除用户
     */
    Boolean delete(Long id);

    /**
     * 更新用户
     */
    Boolean update(User user, Long id);

    /**
     * 获取单个用户
     */
    User getUser(Long id);

    /**
     * 获取用户列表
     */
    List<User> getUser(User user);

}
```
在业务接口的实现中调用了数据库访问接口类(UserDao)的方法实现数据增删改查

## Controller
将请求映射到Service中
```java
@RestController
@Slf4j
public class UserController {
    private final IUserService userService;

    @Autowired
    public UserController(IUserService userService) {
        this.userService = userService;
    }

    @PostMapping("/user")
    public Dict save(@RequestBody User user) {
        Boolean save = userService.save(user);
        return Dict.create().set("code", save ? 200 : 500).set("msg", save ? "成功" : "失败").set("data", save ? user : null);
    }

    @DeleteMapping("/user/{id}")
    public Dict delete(@PathVariable Long id) {
        Boolean delete = userService.delete(id);
        return Dict.create().set("code", delete ? 200 : 500).set("msg", delete ? "成功" : "失败");
    }

    @PutMapping("/user/{id}")
    public Dict update(@RequestBody User user, @PathVariable Long id) {
        Boolean update = userService.update(user, id);
        return Dict.create().set("code", update ? 200 : 500).set("msg", update ? "成功" : "失败").set("data", update ? user : null);
    }

    @GetMapping("/user/{id}")
    public Dict getUser(@PathVariable Long id) {
        User user = userService.getUser(id);
        return Dict.create().set("code", 200).set("msg", "成功").set("data", user);
    }

    @GetMapping("/user")
    public Dict getUser(User user) {
        List<User> userList = userService.getUser(user);
        return Dict.create().set("code", 200).set("msg", "成功").set("data", userList);
    }
}

```


## 测试

先配置mysql，新建springdemo数据库。再配置application.yml。然后运行`data.sql`和`schema.sql`创建表，并往表里插入数据

### 创建数据库表结构
```sql
DROP TABLE IF EXISTS `orm_user`;
CREATE TABLE `orm_user` (
  `id` INT(11) NOT NULL AUTO_INCREMENT PRIMARY KEY COMMENT '主键',
  `name` VARCHAR(32) NOT NULL UNIQUE COMMENT '用户名',
  `password` VARCHAR(32) NOT NULL COMMENT '加密后的密码',
  `salt` VARCHAR(32) NOT NULL COMMENT '加密使用的盐',
  `email` VARCHAR(32) NOT NULL UNIQUE COMMENT '邮箱',
  `phone_number` VARCHAR(15) NOT NULL UNIQUE COMMENT '手机号码',
  `status` INT(2) NOT NULL DEFAULT 1 COMMENT '状态，-1：逻辑删除，0：禁用，1：启用',
  `create_time` DATETIME NOT NULL DEFAULT NOW() COMMENT '创建时间',
  `last_login_time` DATETIME DEFAULT NULL COMMENT '上次登录时间',
  `last_update_time` DATETIME NOT NULL DEFAULT NOW() COMMENT '上次更新时间'
) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='Spring Boot Demo Orm 系列示例表';
```
### 插入数据
```sql
INSERT INTO `orm_user`(`id`,`name`,`password`,`salt`,`email`,`phone_number`) VALUES (1, 'user_1', 'ff342e862e7c3285cdc07e56d6b8973b', '412365a109674b2dbb1981ed561a4c70', 'user1@xkcoding.com', '17300000001');
INSERT INTO `orm_user`(`id`,`name`,`password`,`salt`,`email`,`phone_number`) VALUES (2, 'user_2', '6c6bf02c8d5d3d128f34b1700cb1e32c', 'fcbdd0e8a9404a5585ea4e01d0e4d7a0', 'user2@xkcoding.com', '17300000002');
```

查看数据库信息
![](img/jdbctemplate/1.png)

使用get请求获取用户信息
![](img/jdbctemplate/2.png)

使用Post请求新建用户
![](img/jdbctemplate/3.png)

新建用户后查看数据库表
![](img/jdbctemplate/4.png)